<?php
/*--------------------------------------------------------------------------------------------------
    PropertyQuantityReadService.php 2020-02-18
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

namespace Gambio\Shop\Properties\SellingUnitQuantitiy\Database\Services;

use Gambio\Shop\ProductModifiers\Modifiers\Collections\ModifierIdentifierCollection;
use Gambio\Shop\ProductModifiers\Modifiers\Collections\ModifierIdentifierCollectionInterface;
use Gambio\Shop\Properties\Database\Services\Interfaces\PropertiesReaderServiceInterface;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyModifierIdentifier;
use Gambio\Shop\Properties\Properties\Collections\CombinationCollectionInterface;
use Gambio\Shop\Properties\Properties\Entities\Combination;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Database\Repositories\PropertyQuantityRepositoryInterface;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Entities\AbstractPropertyQuantity;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Entities\CombinationQuantity;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Entities\PropertyTotalQuantity;
use Gambio\Shop\Properties\SellingUnitQuantitiy\Quantitiy\Exceptions\ProductDoesntUsePropertyQuantityException;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;
use ProductDataInterface;

class PropertyQuantityReadService implements PropertyQuantityReadServiceInterface
{
    /**
     * @var bool
     */
    protected $attributeStockCheck;
    /**
     * @var PropertiesReaderServiceInterface
     */
    protected $propertiesReaderService;
    /**
     * @var bool
     */
    protected $stockCheck;


    /**
     * PropertyQuantityReadService constructor.
     *
     * @param PropertiesReaderServiceInterface $propertiesReaderService
     * @param bool $stockCheck
     * @param bool $attributeStockCheck
     */
    public function __construct(
        PropertiesReaderServiceInterface $propertiesReaderService,
        bool $stockCheck,
        bool $attributeStockCheck
    ) {
        $this->propertiesReaderService = $propertiesReaderService;
        $this->stockCheck = $stockCheck;
        $this->attributeStockCheck = $attributeStockCheck;
    }


    /**
     * @inheritDoc
     */
    public function getQuantityBy(SellingUnitId $id, ProductDataInterface $product): ?AbstractPropertyQuantity
    {


        if (($product->getPropertiesQuantityCheck() === 2)
            || ($product->getPropertiesQuantityCheck() === 0 && $this->stockCheck && $this->attributeStockCheck)) {


            $combinations = $this->propertiesReaderService->getCombinationsFor($id);

            if (count($combinations) === 1) {
                return $this->getQuantityByCombination($combinations[0], $product,
                    $this->createModifierIdCollection($id));
            }

            if (count($combinations) > 0) {

                return $this->sumUpQuantities($combinations, $product, $this->createModifierIdCollection($id));
            }
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    public function getQuantityByCombination(
        Combination $combination,
        ProductDataInterface $product,
        ModifierIdentifierCollectionInterface $collection
    ): ?CombinationQuantity {
        try {
            return new CombinationQuantity($combination->quantity()->value(),
                $product->measureUnit(),
                $combination->id(),
                $this->stockCheck,
                $this->attributeStockCheck,
                $product->getPropertiesQuantityCheck(),
                $collection);
        } catch (ProductDoesntUsePropertyQuantityException $e) {
            return null;
        }
    }

    /**
     * @param CombinationCollectionInterface $combinations
     * @param ProductDataInterface $product
     * @param ModifierIdentifierCollectionInterface $collection
     * @return PropertyTotalQuantity|null
     */
    public function sumUpQuantities(
        CombinationCollectionInterface $combinations,
        ProductDataInterface $product,
        ModifierIdentifierCollectionInterface $collection
    ): ?PropertyTotalQuantity {
        if (count($combinations) === 0) {
            return null;
        }

        try {
            $qty = 0;
            foreach ($combinations as $combination) {
                $qty += $combination->quantity()->value();
            }

            return new PropertyTotalQuantity($qty,
                $product->measureUnit(),
                $this->stockCheck,
                $this->attributeStockCheck,
                $product->getPropertiesQuantityCheck(),
                $collection);
        } catch (ProductDoesntUsePropertyQuantityException $e) {
            return null;
        }
    }

    /**
     * @param SellingUnitId $id
     * @return ModifierIdentifierCollectionInterface
     */
    protected function createModifierIdCollection(SellingUnitId $id): ModifierIdentifierCollectionInterface
    {
        $propertyIdList = [];
        foreach ($id->modifiers() as $modifierId) {
            if ($modifierId instanceof PropertyModifierIdentifier) {
                $propertyIdList[] = $modifierId;
            }
        }
        return new ModifierIdentifierCollection($propertyIdList);
    }
}